﻿// Copyright (c) 2012-2015 Sur.ly
// This file is part of Sur.ly SDK.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 3 of the
// License, or (at your option) any later version.
//
// You should have received a copy of the GNU Lesser General Public License
// along with Sur.ly SDK. If not, see http://www.gnu.org/licenses/

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Text;
using System.Web;
using System.Web.Configuration;
using Newtonsoft.Json;

namespace SurlyCore
{
    public static class Surly
    {
        #region Constants

        private const string SettingsToolbarId = "Surly:toolBarId";
        private const string SettingsUseShortUrls = "Surly:useShortUrls";
        private const string SettingsWhiteListUrls = "Surly:whiteListUrls";
        private const string SettingsHtmlParsingEnabled = "Surly:htmlParsingEnabled";

        private const string CacheKeyForShortUrls = "Surly:CachedUrls";
        private const string CacheKeyForWhiteList = "Surly:CachedWhiteList";
        private const string CacheKeyForToolBarId = "Surly:CachedToolBarId";
        private const string CacheKeyForUseShortUrlsFlag = "Surly:CachedKeyForUseShortUrlsFlag";
        private const string CacheKeyForHtmlParsingEnabled = "Surly:CachedKeyForHtmlParsingEnabledFlag";

        private const string Https = "https://";
        private const string Http = "http://";
        private const string BackSlash = "/";
        private const string SurlyUrl = "http://sur.ly/";
        private const string SurlyName = "sur.ly";
        private const string OWithSlash = "/o/";
        private const string Dot = ".";
        private const string Tilda = "~";
        private const string Www = "www";


        #endregion

        public static string ToolBarId
        {
            get
            {
                if (HttpContext.Current.Cache[CacheKeyForToolBarId] == null)
                    HttpContext.Current.Cache[CacheKeyForToolBarId] = String.Empty;

                return (string)HttpContext.Current.Cache[CacheKeyForToolBarId];
            }
            set
            {
                HttpContext.Current.Cache[CacheKeyForToolBarId] = value;
            }
        }
        public static bool UseShortUrls
        {
            get
            {
                if (HttpContext.Current.Cache[CacheKeyForUseShortUrlsFlag] == null)
                    HttpContext.Current.Cache[CacheKeyForUseShortUrlsFlag] = false;

                return (bool)HttpContext.Current.Cache[CacheKeyForUseShortUrlsFlag];
            }
            set
            {
                HttpContext.Current.Cache[CacheKeyForUseShortUrlsFlag] = value;
            }
        }
        private static Dictionary<string, string> CachedShortUrls
        {
            get
            {
                if (HttpContext.Current.Cache[CacheKeyForShortUrls] == null)
                    HttpContext.Current.Cache[CacheKeyForShortUrls] = new Dictionary<string, string>();

                return HttpContext.Current.Cache[CacheKeyForShortUrls] as Dictionary<string, string>;
            }
        }
        public static List<string> WhiteList
        {
            get
            {
                if (HttpContext.Current.Cache[CacheKeyForWhiteList] == null)
                    HttpContext.Current.Cache[CacheKeyForWhiteList] = new List<string>();

                return HttpContext.Current.Cache[CacheKeyForWhiteList] as List<string>;
            }
        }

        public static bool HtmlParsingEnabled
        {
            get
            {
                if (HttpContext.Current.Cache[CacheKeyForHtmlParsingEnabled] == null)
                    HttpContext.Current.Cache[CacheKeyForHtmlParsingEnabled] = false;

                return (bool)HttpContext.Current.Cache[CacheKeyForHtmlParsingEnabled];
            }
            private set
            {
                HttpContext.Current.Cache[CacheKeyForHtmlParsingEnabled] = value;
            }
        }
        static Surly()
        {
            var config = WebConfigurationManager.AppSettings;
            if (config[SettingsToolbarId] != null)
                ToolBarId = config[SettingsToolbarId];

            if (string.IsNullOrEmpty(ToolBarId))
                ToolBarId = "AA000014";

            if (config[SettingsUseShortUrls] != null)
                UseShortUrls = Convert.ToBoolean(config[SettingsUseShortUrls]);

            if (config[SettingsWhiteListUrls] != null)
                WhiteList.AddRange(config[SettingsWhiteListUrls].Split(';'));

            if (config[SettingsHtmlParsingEnabled] != null)
                HtmlParsingEnabled = Convert.ToBoolean(config[SettingsHtmlParsingEnabled]);

            WhiteList.Add(SurlyName);
        }
        public static string ProcessUrl(string link)
        {
            return ProcessUrl(link, ToolBarId);
        }
        public static string ProcessUrl(string link, string toolBarId)
        {
            var domainName = ((string)link.Clone()).ToLower();

            if (!domainName.StartsWith(Https) && !domainName.StartsWith(Http))
                return link;

            if (IsInWhiteList(link))
                return link;

            if (UseShortUrls)
            {
                if (!CachedShortUrls.ContainsKey(link))
                    MakeUrlShort(new[] { link }).ToList().ForEach(item => CachedShortUrls[item.Key] = SurlyUrl + item.Value);

                Debug.WriteLine(CachedShortUrls[link]);

                return CachedShortUrls[link];
            }

            var domainInfo = GetDomainName(domainName);

            domainName = (domainInfo.IsHttps ? Https : Http) + SurlyName + OWithSlash + domainInfo.Name +
                   (String.IsNullOrEmpty(toolBarId) ? String.Empty : BackSlash + toolBarId);

            Debug.WriteLine(domainName);


            return domainName;
        }
        public static Dictionary<string, string> ProcessMultipleUrls(IEnumerable<string> links)
        {
            var resultingLinks = new Dictionary<string, string>();
            var linksToProcessing = links as List<string> ?? links.ToList();

            if (UseShortUrls)
            {
                var tempLinks = new List<string>();
                linksToProcessing.ForEach(link =>
                {
                    if (CachedShortUrls.ContainsKey(link))
                        resultingLinks[link] = CachedShortUrls[link];
                    else if (IsLinkProcessable(link) && !IsInWhiteList(link))
                        tempLinks.Add(link);
                });
                if (tempLinks.Count > 0)
                    MakeUrlShort(tempLinks).ToList().ForEach(item =>
                    {
                        var surlyUrl = SurlyUrl + item.Value;
                        CachedShortUrls[item.Key] = surlyUrl;
                        resultingLinks[item.Key] = surlyUrl;
                    });
            }
            else
            {
                linksToProcessing.ForEach(item =>
                {
                    if (IsLinkProcessable(item))
                        resultingLinks[item] = ProcessUrl(item);
                });
            }

            return resultingLinks;

        }
        public static string Process(string stringToParse)
        {
            var linkFinder = new LinkFinder();
            var links = linkFinder.FindLinks(stringToParse);
            var builder = new StringBuilder(stringToParse);
            var results = ProcessMultipleUrls(links.Select(link => link.Url));

            foreach (var link in links)
            {
                builder.Replace(link.Url, results.ContainsKey(link.Url) ? results[link.Url] : link.Url);
            }
            //links.ForEach(link => builder.Replace(link.Url, results.ContainsKey(link.Url) ? results[link.Url] : link.Url));

            return builder.ToString();
        }
        private static Dictionary<string, string> MakeUrlShort(IEnumerable<string> urls)
        {
            var preparedUrlsable = urls as IList<string> ?? urls.ToList();

            if (preparedUrlsable.ToList().Count == 0)
                throw new ArgumentException("Urls parameter should contains data");

            WebRequest request = WebRequest.Create("http://api.surdotly.com/shorten");
            request.Method = "POST";
            request.ContentType = "application/x-www-form-urlencoded";
            ((HttpWebRequest)request).UserAgent = "surly_api_caller (running " + "sur.ly" + ")";

            byte[] bytes = Encoding.ASCII.GetBytes("raw=1&urls=" + String.Join("\r\n", preparedUrlsable.ToList()));
            request.ContentLength = bytes.Length;
            Stream os = request.GetRequestStream();
            os.Write(bytes, 0, bytes.Length);
            os.Close();

            WebResponse response = request.GetResponse();

            if (response == null)
                throw new WebException("Response from the server is null. Please, check your connection");

            // ReSharper disable once AssignNullToNotNullAttribute
            using (var reader = new StreamReader(response.GetResponseStream()))
            {
                string strResult = reader.ReadToEnd().Trim();
                var resultingUrls = JsonConvert.DeserializeObject<SurlyResponse>(strResult);

                if (resultingUrls.Errors.Count > 0)
                {
                    resultingUrls.Errors.ForEach(error => Trace.WriteLine("asp.net-surly:" + error));
                }

                return resultingUrls.Urls;
            }
        }
        private static bool IsLinkProcessable(string url)
        {
            return !url.StartsWith(BackSlash) && !url.StartsWith(Dot) && !url.StartsWith(Tilda)
                   && (url.ToLower().StartsWith("http") || url.ToLower().StartsWith(Www));
        }

        private static bool IsInWhiteList(string url)
        {
            var tempUrl = GetDomainName(url).Name.ToLower();
            return WhiteList.Any(item => tempUrl.StartsWith(GetDomainName(item).Name.ToLower()));
        }

        private static DomainName GetDomainName(string url)
        {
            var domainName = new DomainName { IsHttps = false, Name = url };

            if (url.StartsWith(Https))
            {
                domainName.IsHttps = true;
                domainName.Name = url.Replace(Https, String.Empty);
            }
            else if (url.StartsWith(Http))
            {
                domainName.Name = url.Replace(Http, String.Empty);
            }

            return domainName;
        }
    }
}