'use strict';

var Surly = (function() {

  /**
   * Constants
   */
  var SURLY_VERSION = '2.2.1',
    SURLY_PANEL_HOST = 'sur.ly',
    SURLY_BACKUP_PANEL_HOST = 'surdotly.com',
    SURLY_DEFAULT_TOOLBAR_ID = 'AA000015',
    DEFAULT_PORTS = {'http': 80, 'https': 443},
    SECURE_PROTOCOL = 'https';

  var msie, /** holds major version number for IE or NaN for real browsers */
    urlParsingNode = document.createElement('a');

  /**
   * Detecting versions of IE
   */
  msie = parseInt((/msie (\d+)/.exec(navigator.userAgent.toLowerCase()) || [])[1], 10);
  if (isNaN(msie)) {
    msie = parseInt((/trident\/.*; rv:(\d+)/.exec(navigator.userAgent.toLowerCase()) || [])[1], 10);
  }

  var isArray = (function() {
    if (typeof Array.isArray !== 'function') {
      return function(value) {
        return Object.prototype.toString.call(value) === '[object Array]';
      };
    }
    return Array.isArray;
  })();

  function isHTMLCollection(nodes) {
    var stringRepr = Object.prototype.toString.call(nodes);

    return /^\[object (HTMLCollection|NodeList)\]$/.test(stringRepr);
  }

  function isElement(node) {
    return !!(node && node.nodeName);
  }

  /**
   * Normalizes and parses a URL
   */
  function urlResolve(url) {
    var href = url;

    if (msie) {
      urlParsingNode.setAttribute('href', href);
      href = urlParsingNode.href;
    }

    urlParsingNode.setAttribute('href', href);

    return {
      href: urlParsingNode.href,
      protocol: urlParsingNode.protocol ? urlParsingNode.protocol.replace(/:$/, '') : '',
      host: urlParsingNode.host,
      search: urlParsingNode.search ? urlParsingNode.search.replace(/^\?/, '') : '',
      hash: urlParsingNode.hash ? urlParsingNode.hash.replace(/^#/, '') : '',
      hostname: urlParsingNode.hostname,
      port: parseInt(urlParsingNode.port) ? parseInt(urlParsingNode.port) : '',
      pathname: (urlParsingNode.pathname.charAt(0) === '/') ? urlParsingNode.pathname : '/' + urlParsingNode.pathname
    };
  }

  function unescapeHTML(str) {
    if (str === null) {
      return '';
    }

    return String(str)
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#039;/g, '\'');
  }

  function rawurlencode(str) {
    str = (str + '');

    return encodeURIComponent(str)
      .replace(/!/g, '%21')
      .replace(/'/g, '%27')
      .replace(/\(/g, '%28')
      .replace(/\)/g, '%29')
      .replace(/\*/g, '%2A');
  }

  /**
   * Surly
   */
  return function(toolbarId) {
    this.toolbarId = toolbarId || SURLY_DEFAULT_TOOLBAR_ID;

    var panelHost = SURLY_PANEL_HOST,
      whiteList = [SURLY_PANEL_HOST, SURLY_BACKUP_PANEL_HOST],
      useCustomPanel = false;

    this.setPanelHost = function(host) {
      if (host && typeof host === 'string') {
        panelHost = host.toLowerCase().replace(/^(https?:\/\/)?(www\d{0,3}\.)?/, '');
        this.whiteList(panelHost);
        useCustomPanel = true;
      }

      return this;
    };

    /**
     * Adds a domain to processing whitelist. Links to whitelisted domains won't be processed
     */
    this.whiteList = function(domain) {
      if (domain && typeof domain === 'string') {
        whiteList.push(domain.toLowerCase().replace(/^(https?:\/\/)?(www\d{0,3}\.)?/, ''));
      }

      return this;
    };

    /**
     * Process Dom element and containing anchor tags
     */
    this.processElement = function(element) {
      if (!isElement(element)) {
        return null;
      }

      var links = element.getElementsByTagName('A');

      for(var i = 0; i < links.length; i++) {
        var href = links[i].getAttribute('href');

        if (href) {
          href = this.processUrl(href);

          if (msie) {
            var text = links[i].innerHTML;
            links[i].setAttribute('href', href);
            if (links[i].innerHTML != text) {
              links[i].innerHTML = text;
            }
          } else {
            links[i].setAttribute('href', href);
          }
        }
      }
    };

    /**
     * Processes an HTML string. Replaces all found links (within <A href>) with links
     * pointing to Sur.ly interstitial page according to initialization parameters
     */
    this.process = function(elements) {
      if (isArray(elements) || isHTMLCollection(elements)) {
        for (var i = 0; i <= elements.length; i++) {
          this.processElement(elements[i]);
        }
      } else {
        this.processElement(elements);
      }
    };

    /**
     * Processes an url. Replaces it with url pointing to Sur.ly interstitial page
     * according to initialization parameters    
     */
    this.processUrl = function(url) {
      if (!/^(https?:\/\/)([^\s\.]+\..+)$/i.test(url)) {
        return url;
      }

      var parsedUrl = urlResolve(unescapeHTML(url));

      if (_isWhitelisted(_normalizeDomain(parsedUrl.hostname))) {
        return url;
      }

      if (parsedUrl.pathname === '/' && parsedUrl.search === '' && parsedUrl.hash === '') {
        parsedUrl.trailingSlash = true;
      }

      return this.composeUrl(parsedUrl);
    };

    /**
     * Processes array of urls. Replaces them with urls pointing to Sur.ly interstitial page
     * according to initialization parameters. Return urls in exactly the same order as receive
     */
    this.processMultipleUrls = function(urls) {
      urls = isArray(urls) ? urls : [];

      for (var i = 0; i < urls.length; i++) {
        urls[i] = this.processUrl(urls[i]);       
      }

      return urls;
    };

    /**
     * Compose url
     */
    this.composeUrl = function(params) {
      return (useCustomPanel ? 'http' : params.protocol) + '://'
        + panelHost
        + _getPrefix(params.protocol)
        + _normalizeDomain(params.hostname)
        + (!params.port ? '' : DEFAULT_PORTS[params.protocol] === params.port ? '' : ':' + params.port)
        + ((params.pathname != '/' || params.search || params.hash) ? '/' : '')
        + (params.pathname != '/' ? rawurlencode(params.pathname.replace(/^\//, '')) : '')
        + (params.search ? rawurlencode('?' + params.search) : '')
        + (params.hash ? rawurlencode('#' + params.hash) : '')
        + (useCustomPanel ? (params.trailingSlash ? '/' : '') : '/' + this.toolbarId);
    };

    function _getPrefix(protocol) {
      if (useCustomPanel) {
        return protocol == SECURE_PROTOCOL ? '/s/' : '/';
      } else {
        return '/o/';
      }
    }

    /**
     * Check whether domain is in whitelist or not
     */
    function _isWhitelisted(domain) {
      domain = domain.toLowerCase() || '';

      for (var i = 0; i < whiteList.length; i++) {
        if (whiteList[i] == domain || '.' + whiteList[i] == domain.slice(-whiteList[i].length -1)) {
          return true;
        }
      }
      return false;
    }

    /**
     * normalize domain
     */
    function _normalizeDomain(domain) {
      return domain.replace(/^www\./i, '');
    }
  };
})();