import unittest
from surly import Surly
import inspect
from test_helper import TestHelper


class ReplaceTest(TestHelper):

    def _get_data(self, method_name, type):
        self.test_name = 'replace'
        return super(ReplaceTest, self)._get_data(method_name, type)

    def setUp(self):
        super(ReplaceTest, self).setUp()
        self.test_host = 'example.com'
        self.toolbar_id = 'AA001753'

    def test_replace_with_toolbar_id(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(toolbar_id=self.toolbar_id, host=self.test_host)
        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'Test strings with toolbar id are not equal'
        )

    def test_remove_url_fragment(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(host=self.test_host)
        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'Remove ulr fragment strings are not equal'
        )

    def test_dont_replace_panel_domains(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(host=self.test_host)
        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'Strings with not replaced panel domains are not equal'
        )

    def test_replace_invalid_urls(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(host=self.test_host)
        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'Invalid urls htmls are not equal'
        )

    def test_replace_invalid_urls_with_toolbar_id(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(toolbar_id=self.toolbar_id, host=self.test_host)
        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'Invalid urls with toolbar ID htmls are not equal'
        )

    def test_dont_replace_relative_links(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(host=self.test_host)
        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'htmls with relative links are not equal'
        )

    def test_domain_whitelist(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(host=self.test_host)

        surly.add_to_whitelist('lE.com')

        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'htmls with whitelisted links are not equal'
        )

    def test_domain_whitelist_with_http(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(host=self.test_host)

        surly.add_to_whitelist('http://test.com')

        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'htmls with relative links are not equal'
        )

    def test_domain_whitelist_with_https(self):
        this_method_name = inspect.stack()[0][3]
        surly = Surly(host=self.test_host)

        surly.add_to_whitelist('https://test.com')

        self.assertEqual(
            surly.process(self._get_data(this_method_name, 'in')),
            self._get_data(this_method_name, 'expected'),
            'htmls with relative links are not equal'
        )

    def test_process_url(self):
        url2expected = {
            'https://www.cnn.com': 'https://sur.ly/o/cnn.com/AA000014',
            'https://cnn.com': 'https://sur.ly/o/cnn.com/AA000014',
            'http://www.cnn.com': 'http://sur.ly/o/cnn.com/AA000014',
            'http://cnn.com': 'http://sur.ly/o/cnn.com/AA000014',

            'https://www.cnn.com/': 'https://sur.ly/o/cnn.com/AA000014',
            'https://cnn.com/': 'https://sur.ly/o/cnn.com/AA000014',
            'http://www.cnn.com/': 'http://sur.ly/o/cnn.com/AA000014',
            'http://cnn.com/': 'http://sur.ly/o/cnn.com/AA000014',

            'https://www.cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA000014',
            'https://cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA000014',
            'http://www.cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA000014',
            'http://cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA000014',

            'https://www.cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',
            'https://cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',
            'http://www.cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',
            'http://cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',

            'http://www.': 'http://www.',
            'https://www.': 'https://www.',
            'http://www-www': 'http://www-www',
            'https://www-www': 'https://www-www',
            'http://www-www.www': 'http://sur.ly/o/www-www.www/AA000014',
            'https://www-www.www': 'https://sur.ly/o/www-www.www/AA000014',
            'http://www.www-www.www': 'http://sur.ly/o/www-www.www/AA000014',
            'https://www.www-www.www': 'https://sur.ly/o/www-www.www/AA000014',
            'http://www': 'http://www',
            'https://www': 'https://www',

            'http://www1.cnn.com': 'http://sur.ly/o/www1.cnn.com/AA000014',
            'https://www1.cnn.com': 'https://sur.ly/o/www1.cnn.com/AA000014',

            'cnn.com': 'cnn.com',
            'www.cnn.com': 'www.cnn.com'
        }

        surly = Surly(host=self.test_host)

        for url, converted_url in url2expected.items():
            self.assertEqual(surly.process_url(url), converted_url, "Url is processed in wrong way")

    def test_process_url_toolbar_id(self):
        url2expected = {
            'https://www.cnn.com': 'https://sur.ly/o/cnn.com/AA001753',
            'https://cnn.com': 'https://sur.ly/o/cnn.com/AA001753',
            'http://www.cnn.com': 'http://sur.ly/o/cnn.com/AA001753',
            'http://cnn.com': 'http://sur.ly/o/cnn.com/AA001753',

            'https://www.cnn.com/': 'https://sur.ly/o/cnn.com/AA001753',
            'https://cnn.com/': 'https://sur.ly/o/cnn.com/AA001753',
            'http://www.cnn.com/': 'http://sur.ly/o/cnn.com/AA001753',
            'http://cnn.com/': 'http://sur.ly/o/cnn.com/AA001753',

            'https://www.cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA001753',
            'https://cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA001753',
            'http://www.cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA001753',
            'http://cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA001753',

            'https://www.cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',
            'https://cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',
            'http://www.cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',
            'http://cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',

            'http://www.': 'http://www.',
            'https://www.': 'https://www.',
            'http://www-www': 'http://www-www',
            'https://www-www': 'https://www-www',
            'http://www-www.www': 'http://sur.ly/o/www-www.www/AA001753',
            'https://www-www.www': 'https://sur.ly/o/www-www.www/AA001753',
            'http://www.www-www.www': 'http://sur.ly/o/www-www.www/AA001753',
            'https://www.www-www.www': 'https://sur.ly/o/www-www.www/AA001753',
            'http://www': 'http://www',
            'https://www': 'https://www',

            'cnn.com': 'cnn.com',
            'www.cnn.com': 'www.cnn.com',
        }

        surly = Surly(self.toolbar_id, host=self.test_host)
        for url, converted_url in url2expected.items():
            self.assertEqual(surly.process_url(url), converted_url, "Url is processed in wrong way. (With toolbar ID)")

    def test_process_multiple_urls(self):
        url2expected = {
            'https://www.cnn.com': 'https://sur.ly/o/cnn.com/AA000014',
            'https://cnn.com': 'https://sur.ly/o/cnn.com/AA000014',
            'http://www.cnn.com': 'http://sur.ly/o/cnn.com/AA000014',
            'http://cnn.com': 'http://sur.ly/o/cnn.com/AA000014',

            'https://www.cnn.com/': 'https://sur.ly/o/cnn.com/AA000014',
            'https://cnn.com/': 'https://sur.ly/o/cnn.com/AA000014',
            'http://www.cnn.com/': 'http://sur.ly/o/cnn.com/AA000014',
            'http://cnn.com/': 'http://sur.ly/o/cnn.com/AA000014',

            'https://www.cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA000014',
            'https://cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA000014',
            'http://www.cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA000014',
            'http://cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA000014',

            'https://www.cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',
            'https://cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',
            'http://www.cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',
            'http://cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',

            'http://www.': 'http://www.',
            'https://www.': 'https://www.',
            'http://www-www': 'http://www-www',
            'https://www-www': 'https://www-www',
            'http://www-www.www': 'http://sur.ly/o/www-www.www/AA000014',
            'https://www-www.www': 'https://sur.ly/o/www-www.www/AA000014',
            'http://www.www-www.www': 'http://sur.ly/o/www-www.www/AA000014',
            'https://www.www-www.www': 'https://sur.ly/o/www-www.www/AA000014',
            'http://www': 'http://www',
            'https://www': 'https://www',

            'cnn.com': 'cnn.com',
            'www.cnn.com': 'www.cnn.com'
        }
        surly = Surly(host=self.test_host)
        for url, converted_url in url2expected.items():
            self.assertEqual(surly.process_multiple_urls([url]), [converted_url], "Urls are processed in wrong way")

        self.assertEqual(
            surly.process_multiple_urls(
                [
                    'https://www.cnn.com/',
                    'https://www.cnn.com/new/ss?ffd',
                    'https://cnn.com/new/ss?ffd'
                ]
            ),
            [
                'https://sur.ly/o/cnn.com/AA000014',
                'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014',
                'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA000014'
            ],
            'Urls passed within the lists do not match'
        )

    def test_multiple_urls_with_toolbar_id(self):
        url2expected = {
            'https://www.cnn.com': 'https://sur.ly/o/cnn.com/AA001753',
            'https://cnn.com': 'https://sur.ly/o/cnn.com/AA001753',
            'http://www.cnn.com': 'http://sur.ly/o/cnn.com/AA001753',
            'http://cnn.com': 'http://sur.ly/o/cnn.com/AA001753',

            'https://www.cnn.com/': 'https://sur.ly/o/cnn.com/AA001753',
            'https://cnn.com/': 'https://sur.ly/o/cnn.com/AA001753',
            'http://www.cnn.com/': 'http://sur.ly/o/cnn.com/AA001753',
            'http://cnn.com/': 'http://sur.ly/o/cnn.com/AA001753',

            'https://www.cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA001753',
            'https://cnn.com/new?ffd': 'https://sur.ly/o/cnn.com/new%3Fffd/AA001753',
            'http://www.cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA001753',
            'http://cnn.com/new?ffd': 'http://sur.ly/o/cnn.com/new%3Fffd/AA001753',

            'https://www.cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',
            'https://cnn.com/new/ss?ffd': 'https://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',
            'http://www.cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',
            'http://cnn.com/new/ss?ffd': 'http://sur.ly/o/cnn.com/new%2Fss%3Fffd/AA001753',

            'http://www.': 'http://www.',
            'https://www.': 'https://www.',
            'http://www-www': 'http://www-www',
            'https://www-www': 'https://www-www',
            'http://www-www.www': 'http://sur.ly/o/www-www.www/AA001753',
            'https://www-www.www': 'https://sur.ly/o/www-www.www/AA001753',
            'http://www.www-www.www': 'http://sur.ly/o/www-www.www/AA001753',
            'https://www.www-www.www': 'https://sur.ly/o/www-www.www/AA001753',
            'http://www': 'http://www',
            'https://www': 'https://www',

            'cnn.com': 'cnn.com',
            'www.cnn.com': 'www.cnn.com'
        }

        surly = Surly(host=self.test_host, toolbar_id=self.toolbar_id)
        for url, converted_url in url2expected.items():
            self.assertEqual(
                surly.process_multiple_urls([url]),
                [converted_url],
                "Urls with toolbar ID are processed in wrong way"
            )

    def test_smart_url_encode(self):
        url2EncodedUrl = {
            'https://www.cnn.com': 'cnn.com',
            'https://cnn.com': 'cnn.com',
            'http://www.cnn.com': 'cnn.com',
            'http://cnn.com': 'cnn.com',
            'www.cnn.com': 'cnn.com',
            'cnn.com': 'cnn.com',

            'https://www.cnn.com/': 'cnn.com',
            'https://cnn.com/': 'cnn.com',
            'http://www.cnn.com/': 'cnn.com',
            'http://cnn.com/': 'cnn.com',
            'www.cnn.com/': 'cnn.com',
            'cnn.com/': 'cnn.com',

            'https://www.cnn.com/new?ffd': 'cnn.com/new%3Fffd',
            'https://cnn.com/new?ffd': 'cnn.com/new%3Fffd',
            'http://www.cnn.com/new?ffd': 'cnn.com/new%3Fffd',
            'http://cnn.com/new?ffd': 'cnn.com/new%3Fffd',
            'www.cnn.com/new?ffd': 'cnn.com/new%3Fffd',
            'cnn.com/new?ffd': 'cnn.com/new%3Fffd',

            'https://www.cnn.com/new/ss?ffd': 'cnn.com/new%2Fss%3Fffd',
            'https://cnn.com/new/ss?ffd': 'cnn.com/new%2Fss%3Fffd',
            'http://www.cnn.com/new/ss?ffd': 'cnn.com/new%2Fss%3Fffd',
            'http://cnn.com/new/ss?ffd': 'cnn.com/new%2Fss%3Fffd',
            'www.cnn.com/new/ss?ffd': 'cnn.com/new%2Fss%3Fffd',
            'cnn.com/new/ss?ffd': 'cnn.com/new%2Fss%3Fffd',

            '//////': '%2F%2F%2F%2F%2F%2F',
            'http://www.': 'www.',
            'https://www.': 'www.'
        }

        surly = Surly(host=self.test_host)
        for url, encoded_url in url2EncodedUrl.items():
            self.assertEqual(surly._smart_url_encode(url), encoded_url, "Url is not encoded properly")


if __name__ == '__main__':
    unittest.main()


